"""Repository for EmailLog records."""

from __future__ import annotations

from datetime import datetime, date
from typing import List
from tinydb import Query

from ca_task_manager.models.email_log import EmailLog
from ca_task_manager.repositories.base import TinyDBRepository


class EmailLogRepository(TinyDBRepository):
    def __init__(self, db_path: str = "data/email_logs.json"):
        super().__init__(db_path, table_name="email_logs", model_class=EmailLog)

    def create_log(self, log: EmailLog) -> int:
        return self.insert(log.dict())

    def logs_for_employee(self, employee_id: int) -> List[dict]:
        return self.filter(employee_id=employee_id)

    def logs_for_date(self, day: date) -> List[dict]:
        """Return logs sent on a given date."""
        iso_prefix = day.isoformat()
        # Filter logs where sent_at date matches day
        results: List[dict] = []
        for item in self.table.all():
            sent_at = item.get("sent_at")
            if not sent_at:
                continue
            if isinstance(sent_at, str):
                # Expect ISO format
                if sent_at.startswith(iso_prefix):
                    results.append(item)
            elif isinstance(sent_at, datetime):
                if sent_at.date() == day:
                    results.append(item)
        return results